/*jshint esversion: 6 */

define(["lodash", "src/utils", 
	"src/math/Mat3", "src/math/Vec2", "src/math/mathUtils",
	"lib/tasks/dofs", "lib/tasks/handle", "lib/tasks/puppet", "lib/tasks/layer",
	],
function(lodash, utils,
	mat3, vec2, mathUtils,
	dofs, handle, puppet, layer) {
"use strict";

var exports = {
	dofs,
	handle,
	puppet,
	layer
};


var aDofKeys = lodash(dofs.TransformDefault).omit("type").keys().value();

/*============================================================================
	Tasks
============================================================================*/

function MoveTo (params) {
	this.tDof = lodash.pick(params, aDofKeys);
}
utils.mixin(MoveTo, {
	update (/* prev */) {
		return lodash.assign({}, this.tDof);
	},
	tag () {
		return "tLayerTaskItems";
	}
});
exports.MoveTo = MoveTo;


// add or initialize with delta 
function sumWith (delta) {
	return function (val) {
		if (lodash.isUndefined(val)) {
			return delta;
		} else {
			return val + delta;
		}
	};
}

// multiply or initialize with delta 
function multiplyWith (delta) {
	return function (val) {
		if (lodash.isUndefined(val)) {
			return delta;
		} else {
			return val * delta;
		}
	};
}

var mathCos = Math.cos,
	mathSin = Math.sin;

function MoveBy (params) {
	this.tDof = lodash.pick(params, aDofKeys);
}
utils.mixin(MoveBy, {
	update (prev) {
		var tDof = this.tDof;

		var typePrev = prev.type,
			typeNow = typePrev;

		var result = lodash.assign({}, prev);
		lodash.forOwn(tDof, function (delta, key) {
			var ds, dx, dy, anglePrev;

			switch (key) {
				case "x":
					typeNow = dofs.type.merge(typeNow, dofs.type.kTranslation);


					// without previous rotation only x changes
					anglePrev = prev.angle;
					if (anglePrev === 0) {
						lodash.update(result, key, sumWith(delta));
					} else {
						// otherwise both x and y change
						ds = prev.xScale * delta;
						dx = mathCos(anglePrev) * ds;
						dy = mathSin(anglePrev) * ds;

						lodash.update(result, key, sumWith(dx));
						lodash.update(result, "y", sumWith(dy));
					}

				break;

				case "y":
					typeNow = dofs.type.merge(typeNow, dofs.type.kTranslation);

					// without previous rotation only y changes
					anglePrev = prev.angle;
					if (anglePrev === 0) {
						lodash.update(result, key, sumWith(delta));
					} else {
						// otherwise both x and y change
						ds = prev.yScale * delta;
						dx = -mathSin(anglePrev) * ds;
						dy = mathCos(anglePrev) * ds;

						lodash.update(result, key, sumWith(dy));
						lodash.update(result, "x", sumWith(dx));
					}

				break;

				case "xScale":

					typeNow = dofs.type.merge(typeNow, dofs.type.kLinear);
					lodash.update(result, key, multiplyWith(delta));

				break;

				case "yScale":

					typeNow = dofs.type.merge(typeNow, dofs.type.kLinear);
					lodash.update(result, key, multiplyWith(delta));

				break;

				case "angle":

					typeNow = dofs.type.merge(typeNow, dofs.type.kLinear);
					lodash.update(result, key, sumWith(delta));

				break;

				case "xShear":

					typeNow = dofs.type.merge(typeNow, dofs.type.kLinear);
					lodash.update(result, key, sumWith(delta));

				break;
			}
		});

		result.type = typeNow;
		return result;
	},
	tag () {
		return "tLayerPostWarpTaskItems";
	}


});
exports.MoveBy = MoveBy;


function differenceFromMatrix (matFrom, transformTo)
{
	var position = [], scale = [], aShear = [], aRotation = [];

	mat3.decomposeAffine(matFrom, position, scale, aShear, aRotation);

	return {
		x : transformTo.x - position[0],
		y : transformTo.y - position[1],
		angle : transformTo.angle - mathUtils.angle(aRotation),
		xScale : transformTo.xScale / scale[0],
		yScale : transformTo.yScale / scale[1]
	};
}
exports.differenceFromMatrix = differenceFromMatrix;


return exports;

}); // end define